# -*- coding: utf-8 -*-
"""属性装饰器和高级特性示例"""

print("=== 测试1: @property装饰器 ===")
class Temperature:
    """温度类 - 演示属性装饰器"""
    def __init__(self, celsius=0):
        self._celsius = celsius
    
    @property
    def celsius(self):
        """摄氏度属性的getter"""
        print("获取摄氏度")
        return self._celsius
    
    @celsius.setter
    def celsius(self, value):
        """摄氏度属性的setter"""
        print(f"设置摄氏度为 {value}")
        if value < -273.15:
            raise ValueError("温度不能低于绝对零度(-273.15°C)")
        self._celsius = value
    
    @property
    def fahrenheit(self):
        """华氏度属性（只读）"""
        return self._celsius * 9/5 + 32
    
    @property
    def kelvin(self):
        """开尔文温度属性（只读）"""
        return self._celsius + 273.15

# 使用属性装饰器
temp = Temperature(25)
print(f"初始温度: {temp.celsius}°C")
print(f"华氏度: {temp.fahrenheit}°F")
print(f"开尔文: {temp.kelvin}K")

print("\n修改温度:")
temp.celsius = 30
print(f"新温度: {temp.celsius}°C")
print(f"华氏度: {temp.fahrenheit}°F")

print("\n尝试设置无效温度:")
try:
    temp.celsius = -300
except ValueError as e:
    print(f"错误: {e}")

print("\n=== 测试2: 描述符协议 ===")
class Descriptor:
    """描述符类"""
    def __init__(self, name):
        self.name = name
    
    def __get__(self, obj, objtype=None):
        print(f"获取属性 {self.name}")
        return obj.__dict__.get(self.name, None)
    
    def __set__(self, obj, value):
        print(f"设置属性 {self.name} = {value}")
        if not isinstance(value, str):
            raise TypeError(f"{self.name} 必须是字符串")
        obj.__dict__[self.name] = value
    
    def __delete__(self, obj):
        print(f"删除属性 {self.name}")
        del obj.__dict__[self.name]

class Person:
    """使用描述符的人员类"""
    name = Descriptor('name')
    
    def __init__(self, name):
        self.name = name

person = Person("张三")
print(f"姓名: {person.name}")
person.name = "李四"
print(f"新姓名: {person.name}")

try:
    person.name = 123
except TypeError as e:
    print(f"错误: {e}")

print("\n=== 测试3: 类装饰器 ===")
def add_str_method(cls):
    """类装饰器 - 为类添加__str__方法"""
    def __str__(self):
        attrs = ', '.join(f'{k}={v}' for k, v in self.__dict__.items())
        return f'{cls.__name__}({attrs})'
    
    cls.__str__ = __str__
    return cls

@add_str_method
class Student:
    """学生类 - 使用类装饰器"""
    def __init__(self, name, age, grade):
        self.name = name
        self.age = age
        self.grade = grade

student = Student("王五", 18, "高三")
print(f"学生信息: {student}")

print("\n=== 测试4: 数据类（Python 3.7+）===")
try:
    from dataclasses import dataclass, field
    from typing import List
    
    @dataclass
    class Book:
        """书籍数据类"""
        title: str
        author: str
        price: float = 0.0
        tags: List[str] = field(default_factory=list)
        
        def __post_init__(self):
            """初始化后处理"""
            if self.price < 0:
                self.price = 0.0
    
    book1 = Book("Python编程", "作者A", 59.9, ["编程", "Python"])
    book2 = Book("数据结构", "作者B")
    
    print(f"书籍1: {book1}")
    print(f"书籍2: {book2}")
    print(f"书籍相等性: {book1 == book2}")
    
except ImportError:
    print("当前Python版本不支持dataclass")

print("\n=== 测试5: 抽象基类 ===")
from abc import ABC, abstractmethod

class Shape(ABC):
    """抽象形状类"""
    @abstractmethod
    def area(self):
        """计算面积 - 抽象方法"""
        pass
    
    @abstractmethod
    def perimeter(self):
        """计算周长 - 抽象方法"""
        pass
    
    def description(self):
        """具体方法"""
        return f"这是一个{self.__class__.__name__}"

class Rectangle(Shape):
    """矩形类 - 实现抽象基类"""
    def __init__(self, width, height):
        self.width = width
        self.height = height
    
    def area(self):
        return self.width * self.height
    
    def perimeter(self):
        return 2 * (self.width + self.height)

class Circle(Shape):
    """圆形类 - 实现抽象基类"""
    def __init__(self, radius):
        self.radius = radius
    
    def area(self):
        return 3.14159 * self.radius ** 2
    
    def perimeter(self):
        return 2 * 3.14159 * self.radius

# 不能直接实例化抽象类
try:
    shape = Shape()
except TypeError as e:
    print(f"无法实例化抽象类: {e}")

# 实例化具体类
rect = Rectangle(5, 3)
circle = Circle(4)

print(f"\n矩形: {rect.description()}")
print(f"面积: {rect.area()}, 周长: {rect.perimeter()}")

print(f"\n圆形: {circle.description()}")
print(f"面积: {circle.area():.2f}, 周长: {circle.perimeter():.2f}")

print("\n程序结束")