#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
迭代器演示代码
"""

# 自定义迭代器类
class CountDown:
    """
    倒计时迭代器
    """
    def __init__(self, start):
        self.start = start
    
    def __iter__(self):
        return self
    
    def __next__(self):
        if self.start <= 0:
            raise StopIteration
        self.start -= 1
        return self.start + 1

print("=== 自定义迭代器类 ===")
countdown = CountDown(5)
print("倒计时:")
for num in countdown:
    print(f"  {num}")

# 可迭代对象 vs 迭代器
print("\n=== 可迭代对象 vs 迭代器 ===")
my_list = [1, 2, 3, 4, 5]
print(f"列表是否可迭代: {hasattr(my_list, '__iter__')}")
print(f"列表是否是迭代器: {hasattr(my_list, '__next__')}")

# 获取迭代器
list_iterator = iter(my_list)
print(f"列表迭代器是否可迭代: {hasattr(list_iterator, '__iter__')}")
print(f"列表迭代器是否是迭代器: {hasattr(list_iterator, '__next__')}")

print("使用迭代器逐个获取元素:")
print(f"  {next(list_iterator)}")
print(f"  {next(list_iterator)}")
print(f"  {next(list_iterator)}")

# 斐波那契迭代器
class FibonacciIterator:
    """
    斐波那契数列迭代器
    """
    def __init__(self, max_count):
        self.max_count = max_count
        self.count = 0
        self.a, self.b = 0, 1
    
    def __iter__(self):
        return self
    
    def __next__(self):
        if self.count >= self.max_count:
            raise StopIteration
        
        if self.count == 0:
            self.count += 1
            return self.a
        elif self.count == 1:
            self.count += 1
            return self.b
        else:
            self.a, self.b = self.b, self.a + self.b
            self.count += 1
            return self.b

print("\n=== 斐波那契迭代器 ===")
fib_iter = FibonacciIterator(8)
fib_numbers = list(fib_iter)
print(f"前8个斐波那契数: {fib_numbers}")

# 内置迭代器工具
print("\n=== 内置迭代器工具 ===")

# enumerate函数
fruits = ['apple', 'banana', 'cherry']
print("enumerate函数:")
for index, fruit in enumerate(fruits, start=1):
    print(f"  {index}: {fruit}")

# zip函数
names = ['Alice', 'Bob', 'Charlie']
ages = [25, 30, 35]
print("\nzip函数:")
for name, age in zip(names, ages):
    print(f"  {name} is {age} years old")

# reversed函数
print("\nreversed函数:")
for fruit in reversed(fruits):
    print(f"  {fruit}")

# 迭代器的状态管理
class StatefulIterator:
    """
    有状态的迭代器示例
    """
    def __init__(self, data):
        self.data = data
        self.index = 0
        self.access_count = 0
    
    def __iter__(self):
        return self
    
    def __next__(self):
        if self.index >= len(self.data):
            raise StopIteration
        
        value = self.data[self.index]
        self.index += 1
        self.access_count += 1
        return value
    
    def get_access_count(self):
        return self.access_count

print("\n=== 迭代器的状态管理 ===")
stateful_iter = StatefulIterator(['a', 'b', 'c', 'd'])
print("迭代过程:")
for item in stateful_iter:
    print(f"  当前元素: {item}, 访问次数: {stateful_iter.get_access_count()}")

# 迭代器链
class ChainIterator:
    """
    链式迭代器，可以连接多个可迭代对象
    """
    def __init__(self, *iterables):
        self.iterables = iterables
        self.current_iterable = 0
        self.current_iterator = iter(self.iterables[0]) if self.iterables else iter([])
    
    def __iter__(self):
        return self
    
    def __next__(self):
        while self.current_iterable < len(self.iterables):
            try:
                return next(self.current_iterator)
            except StopIteration:
                self.current_iterable += 1
                if self.current_iterable < len(self.iterables):
                    self.current_iterator = iter(self.iterables[self.current_iterable])
                else:
                    raise StopIteration
        raise StopIteration

print("\n=== 迭代器链 ===")
chain_iter = ChainIterator([1, 2, 3], ['a', 'b'], [10, 20])
chained_result = list(chain_iter)
print(f"链式迭代结果: {chained_result}")

# itertools模块简单示例
import itertools

print("\n=== itertools模块示例 ===")
# itertools.count - 无限计数器
print("itertools.count (前5个):")
counter = itertools.count(10, 2)  # 从10开始，步长为2
for i, value in enumerate(counter):
    if i >= 5:
        break
    print(f"  {value}")

# itertools.cycle - 循环迭代
print("\nitertools.cycle (前8个):")
cycler = itertools.cycle(['A', 'B', 'C'])
for i, value in enumerate(cycler):
    if i >= 8:
        break
    print(f"  {value}")

# itertools.chain - 连接迭代器
print("\nitertools.chain:")
chained = itertools.chain([1, 2, 3], ['x', 'y'], [100])
print(f"  {list(chained)}")